<?php

namespace App\Http\Controllers;

use App\Models\Barang;
use App\Models\Penjualan;
use App\Models\DetailPenjualan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PenjualanController extends Controller
{
    public function index(Request $request)
    {
        $isMobile = $request->header('User-Agent') &&
            preg_match('/Mobile|Android|iPhone|iPad/i', $request->header('User-Agent'));

        $perPage = $isMobile ? 6 : 10;

        $query = Penjualan::query();

        // 🔍 SEARCH CUSTOMER
        if ($request->filled('customer')) {
            $query->where('customer_name', 'like', '%' . $request->customer . '%');
        }

        // 📅 FILTER TANGGAL
        if ($request->filled('tanggal')) {
            $query->whereDate('tanggal', $request->tanggal);
        }

        $penjualan = $query
            ->orderBy('tanggal', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate($perPage)
            ->withQueryString();

        // 📊 SUMMARY BULANAN
        $summaryBulanan = Penjualan::select(
            DB::raw("DATE_FORMAT(tanggal, '%Y-%m') as bulan"),
            DB::raw('COUNT(*) as total_transaksi'),
            DB::raw('SUM(total_harga) as total_nominal')
        )
            ->groupBy('bulan')
            ->orderBy('bulan', 'desc')
            ->get();

        // 💰 TOTAL KESELURUHAN
        $totalKeseluruhan = Penjualan::sum('total_harga');

        // 🫙 TOTAL QTY TERJUAL (INI YANG KURANG)
        $totalQty = DetailPenjualan::whereHas('penjualan', function ($q) use ($request) {

            if ($request->filled('customer')) {
                $q->where('customer_name', 'like', '%' . $request->customer . '%');
            }

            if ($request->filled('tanggal')) {
                $q->whereDate('tanggal', $request->tanggal);
            }
        })->sum('qty');

        return view('penjualan.index', compact(
            'penjualan',
            'summaryBulanan',
            'totalKeseluruhan',
            'totalQty' // ⬅️ WAJIB
        ));
    }

    public function create()
    {
        $barang = Barang::where('status', true)->get();
        return view('penjualan.create', compact('barang'));
    }

    public function edit(Penjualan $penjualan)
    {
        $penjualan->load('details.barang');

        // Ambil barang yang statusnya aktif
        $barang = Barang::where('status', true)->get();

        // Ambil ID barang yang sudah dipilih di penjualan ini
        $selectedBarangIds = $penjualan->details->pluck('barang_id')->toArray();

        // Ambil barang yang sudah dipilih tapi mungkin statusnya false
        $selectedBarang = Barang::whereIn('id', $selectedBarangIds)
            ->where('status', false)
            ->get();

        // Gabungkan barang aktif dengan barang yang sudah dipilih (tapi non-aktif)
        $barang = $barang->merge($selectedBarang)->unique('id')->values();

        return view('penjualan.edit', compact('penjualan', 'barang'));
    }

    public function show(Penjualan $penjualan)
    {
        $penjualan->load(['details.barang', 'user']);

        return view('penjualan.show', compact('penjualan'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_name' => 'required|string|max:100',
            'tanggal' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.barang_id' => 'required|exists:barang,id',
            'items.*.qty' => 'required|integer|min:1',
        ]);

        DB::transaction(function () use ($validated) {

            $penjualan = Penjualan::create([
                'invoice' => 'INV-' . now()->format('Ymd') . '-' . Str::upper(Str::random(5)),
                'customer_name' => $validated['customer_name'], // ✅
                'tanggal' => $validated['tanggal'],
                'status' => 'diproses',
                'user_id' => auth()->id(),
                'total_harga' => 0,
            ]);

            $total = 0;

            foreach ($validated['items'] as $item) {
                $barang = Barang::findOrFail($item['barang_id']);

                $subtotal = $barang->harga * $item['qty'];

                DetailPenjualan::create([
                    'penjualan_id' => $penjualan->id,
                    'barang_id' => $barang->id,
                    'qty' => $item['qty'],
                    'harga' => $barang->harga,
                    'subtotal' => $subtotal,
                ]);

                $total += $subtotal;
            }

            $penjualan->update([
                'total_harga' => $total
            ]);
        });

        return redirect()
            ->route('penjualan.index')
            ->with('success', 'Penjualan berhasil disimpan');
    }

    public function update(Request $request, Penjualan $penjualan)
    {
        $request->validate([
            'tanggal' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.barang_id' => 'required|exists:barang,id',
            'items.*.qty' => 'required|integer|min:1',
        ]);

        DB::transaction(function () use ($request, $penjualan) {

            // 1️⃣ Update header penjualan
            $penjualan->update([
                'tanggal' => $request->tanggal,
            ]);

            // 2️⃣ Hapus semua detail lama (AMAN & SEDERHANA)
            $penjualan->details()->delete();

            $total = 0;

            // 3️⃣ Insert ulang detail
            foreach ($request->items as $item) {
                $barang = Barang::find($item['barang_id']);
                $subtotal = $barang->harga * $item['qty'];

                DetailPenjualan::create([
                    'penjualan_id' => $penjualan->id,
                    'barang_id' => $barang->id,
                    'qty' => $item['qty'],
                    'harga' => $barang->harga,
                    'subtotal' => $subtotal,
                ]);

                $total += $subtotal;
            }

            // 4️⃣ Update total harga
            $penjualan->update([
                'total_harga' => $total,
            ]);
        });

        return redirect()->route('penjualan.index')
            ->with('success', 'Penjualan berhasil diperbarui');
    }

    public function destroy(Penjualan $penjualan)
    {
        DB::transaction(function () use ($penjualan) {
            // Hapus semua detail penjualan
            $penjualan->details()->delete();

            // Hapus penjualan
            $penjualan->delete();
        });

        return redirect()->route('penjualan.index')
            ->with('success', 'Penjualan berhasil dihapus');
    }

    public function updateStatus(Request $request, Penjualan $penjualan)
    {
        $request->validate([
            'status' => 'required|in:draft,menunggu_pembayaran,diproses'
        ]);

        $penjualan->update([
            'status' => $request->status
        ]);

        return back()->with('success', 'Status penjualan diperbarui');
    }
}
